<?php

namespace App\Http\Controllers;

use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;

use App\Models\PriceCountry;
use Illuminate\Http\Request;
use App\Http\Resources\PriceCountry as PriceCountryCollection;

class PriceCountryController extends Controller
{
    public function index(Request $request)
        {
            try {

                $limit = $request->query('limit', 10);

                $query = PriceCountry::query();
                // Get all parameters from the request
                $parameters = $request->all();

            // Loop through each parameter and apply filters to the query
                foreach ($parameters as $key => $value) {
                    // Make sure the key is not a reserved Laravel parameter
                    if ($key !== 'page' && $key !== 'limit') {
                        if (in_array($key, ['country_name'])) {
                            // Apply search functionality using 'like' for name, email, and phone
                            $query->where($key, 'like', "%$value%");
                        } else {
                            // Apply exact match for other fields
                            $query->where($key, $value);
                        }
                    }
                }

                $pricecountrys = $query->orderBy('id')->paginate($limit);
                $pricecountryCollection = PriceCountryCollection::collection($pricecountrys->items());

                $responseData = [
                    'records' => $pricecountryCollection,
                    'pagination' => [
                        'current_page' => $pricecountrys->currentPage(),
                        'last_page' => $pricecountrys->lastPage(),
                        'per_page' => $pricecountrys->perPage(),
                        'total' => $pricecountrys->total(),
                    ],
                ];
                
                return response()->json($responseData);
            } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
    
        /**
         * Store a newly created resource in storage.
         */
        public function store(Request $request)
        {
            try {
                $validatedData = $request->validate([
                    'country_name' => 'required',
                ]);
    
                DB::beginTransaction();
                $pricecountry = PriceCountry::create([
                    'country_name' => $request->country_name,
                    'status' => $request->status ?? true,
                ]);
     
                DB::commit();
                return response()->json(['message' => 'Country created successfully', 'record' => new PriceCountryCollection($pricecountry)], 201);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Display the specified resource.
         */
        public function show($id)
        {
            try{
                $pricecountry = PriceCountry::findOrFail($id);
                return response()->json(['message' => 'Country Details Fetached successfully', 'record' => new PriceCountryCollection($pricecountry)], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
            
        }
    
    
        /**
         * Update the specified resource in storage.
         */
        public function update(Request $request, $id)
        {
            try {
                $validatedData = $request->validate([
                    'country_name' => 'required',
                ]);
    
                DB::beginTransaction();
        
                $pricecountry = PriceCountry::findOrFail($id); 
                // Update user data

                $pricecountry->country_name = $request->country_name;
                $pricecountry->status = $request->status;
                $pricecountry->save();
    
                DB::commit();
        
                return response()->json(['message' => 'Country updated successfully', 'record' => new PriceCountryCollection($pricecountry)], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'PriceCountry not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Remove the specified resource from storage.
         */
        public function destroy($id)
        {
            try {

                $ids = explode(",",$id);
                if (!empty($ids)) {
                    // Fetch the pricecountry  with the specified IDs
                    $pricecountrys = PriceCountry::whereIn('id', $ids)->get();
                    
                    // Delete the fetched pricecountry groups
                    foreach ($pricecountrys as $pricecountry) {
                        $pricecountry->delete();
                    }
                }
                return response()->json(['message' => 'Country and associated entries deleted successfully'], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Country not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }


        // Other Resources

        public function activePriceCountry()
        {
            try {
                $pricecountry = PriceCountry::where('status',true)->orderBy('id')->get();
    
                return response()->json(['records' => PriceCountryCollection::collection($pricecountry)]);
            } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    }