<?php

namespace App\Http\Controllers;
use App\Models\User;
use App\Models\Manager;
use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\Manager as ManagerCollection;

class ManagerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            
            $limit = $request->query('limit', 10);




            $query = User::query();
            // Get all parameters from the request
            $parameters = $request->all();

            // Loop through each parameter and apply filters to the query
            foreach ($parameters as $key => $value) {
                // Make sure the key is not a reserved Laravel parameter
                if ($key !== 'page' && $key !== 'limit') {
                    if (in_array($key, ['name', 'email','phone'])) {
                        // Apply search functionality using 'like' for name, email, and phone
                        $query->where($key, 'like', "%$value%");
                    } else {
                        // Apply exact match for other fields
                        $query->where($key, $value);
                    }
                }
            }

            $users = $query->where('role','manager')->orderBy('updated_at', 'desc')->paginate($limit);
            $userCollection = ManagerCollection::collection($users->items());
            $responseData = [
                'records' => $userCollection,
                'pagination' => [
                    'current_page' => $users->currentPage(),
                    'last_page' => $users->lastPage(),
                    'per_page' => $users->perPage(),
                    'total' => $users->total(),
                ],
            ];

            return response()->json($responseData);
        } catch (QueryException $e) {
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'name' => 'required|string|max:100',
                'email' => 'required|unique:users|max:255',
                //'phone' => 'required|unique:users',
                //'password' => 'sometimes|nullable|string|min:8|confirmed',
            ]);

            DB::beginTransaction();

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'role' => 'manager',
                'password' => bcrypt($request->password),
                'status' => $request->status ?? true, // Default to true if status is not provided.
            ]);
 
            // Create the manager for the created user
            $manager = Manager::create([
                'user_id' => $user->id,
                //'alc_list' => 'nothing', // Default to null if alc list is not provided.
                'profile_pic_id' => $request->profile_pic_id
            ]);

            // Commit the transaction
            DB::commit();
            return response()->json(['message' => 'Manager created successfully', 'record' => new ManagerCollection($user)], 201);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try{
            $user = User::findOrFail($id);
            return response()->json(['message' => 'Manager Details Fetached successfully', 'record' => new ManagerCollection($user)], 200);
        }catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
        
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $validatedData = $request->validate([
                'name' => 'required|string|max:100',
                'email' => 'required|unique:users,email,' . $id . '|max:255',
                //'phone' => 'sometimes|required|unique:users,phone,' . $id,
                //'password' => 'sometimes|nullable|string|min:8|confirmed',
            ]);

            DB::beginTransaction();
    
            $user = User::findOrFail($id);
            $manager = Manager::where('user_id', $user->id)->firstOrFail();
    
            // Update user data
            $user->name = $request->name;
            $user->email = $request->email;
            if ($request->has('phone')) {
                $user->phone = $request->phone;
            }
            if ($request->filled('password')) {
                $user->password = bcrypt($request->password);
            }
            $user->status = $request->status ?? $user->status;
            $user->save();

            DB::commit();
    
            return response()->json(['message' => 'Manager updated successfully', 'record' => new ManagerCollection($user)], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'User or Manager not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $user = User::findOrFail($id);
            $user->delete();
            return response()->json(['message' => 'Manager and associated user deleted successfully'], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'User or Manager not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    // Profile Pic
    public function profilePic(Request $request){
        try {
            DB::beginTransaction();

            $manager = Manager::where('user_id', $request->id)->firstOrFail();
            $manager->profile_pic_id = $request->profile_pic_id ?? $manager->profile_pic_id;
            $manager->save();
            $user = User::findOrFail($request->id);
            DB::commit();
            return response()->json(['message' => 'Profile picture is updated successfully', 'record' => new ManagerCollection($user)], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'User or Customer not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }

    public function roleUpdate(Request $request, $id)
    {
        try {

            DB::beginTransaction();
    
            $manager = Manager::where('user_id',$id)->firstOrFail();
            $manager->role_id = $request->role_id;
            $manager->save();

            DB::commit();
    
            return response()->json(['message' => 'User role updated successfully'], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['message' => 'User or Manager not found'], 404);
        } catch (QueryException $e) {
            DB::rollBack();
            return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
        }
    }
}
