<?php

namespace App\Http\Controllers;
use App\Models\User;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\Customer as CustomerCollection;
use Illuminate\Validation\ValidationException;


class CustomerController extends Controller{
        /**
         * Display a listing of the resource.
         */
        public function index(Request $request)
        {
            try {


                $limit = $request->query('limit', 10);




                $query = User::query();
                // Get all parameters from the request
                $parameters = $request->all();
    
                // Loop through each parameter and apply filters to the query
                foreach ($parameters as $key => $value) {
                    // Make sure the key is not a reserved Laravel parameter
                    if ($key !== 'page' && $key !== 'limit' && $key !== 'date_range') {
                        if (in_array($key, ['name', 'email','phone'])) {
                            // Apply search functionality using 'like' for name, email, and phone
                            $query->where($key, 'like', "%$value%");
                        } else {
                            // Apply exact match for other fields
                            $query->where($key, $value);
                        }
                    }
                }

                // Apply date range filter
                if ($request->has('date_range')) {
                    $dates = explode(' - ', $request->input('date_range'));
                    $startDate = \Carbon\Carbon::createFromFormat('d/m/Y', $dates[0])->startOfDay();
                    $endDate = \Carbon\Carbon::createFromFormat('d/m/Y', $dates[1])->endOfDay();
                    $query->whereBetween('created_at', [$startDate, $endDate]);
                }
    
                $users = $query->where('role','customer')->orderBy('updated_at', 'desc')->paginate($limit);
                $userCollection = CustomerCollection::collection($users->items());
                $responseData = [
                    'records' => $userCollection,
                    'pagination' => [
                        'current_page' => $users->currentPage(),
                        'last_page' => $users->lastPage(),
                        'per_page' => $users->perPage(),
                        'total' => $users->total(),
                    ],
                ];
    
                return response()->json($responseData);
            } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
    
        /**
         * Store a newly created resource in storage.
         */
        public function store(Request $request)
        {
            try {
                $validatedData = $request->validate([
                    'name' => 'required|string|max:100',
                    'email' => 'required|unique:users|max:255',
                    'phone' => 'required|unique:users',
                    //'password' => 'required|string|min:8|confirmed', 
                ]);
    
                DB::beginTransaction();
    
                $user = User::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'phone' => $request->phone,
                    'role' => 'customer',
                    'password' => bcrypt($request->password),
                    'status' => $request->status ?? true, // Default to true if status is not provided.
                ]);
     
                // Create the customer for the created user
                $customer = Customer::create([
                    'user_id' => $user->id,
                    'cover_pic_id' => $request->cover_pic_id ?? null, //if cover_pic_id not assigned it will set to null
                    'profile_pic_id' => $request->profile_pic_id ?? null, //if profile_pic_id not assigned it will set to null
                    'company_name' => $request->company_name,
                    'whatsapp_number' => $request->whatsapp_number,
                    'country_id' => $request->country_id,
                    'state_id' => $request->state_id,
                    'city_id' => $request->city_id,
                    'address' => $request->address,
                    'pincode' => $request->pincode,
                ]);
    
                // Commit the transaction
                DB::commit();
                return response()->json(['message' => 'Customer created successfully', 'record' => new CustomerCollection($user)], 201);
           } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Display the specified resource.
         */
        public function show($id)
        {
            try{
                $user = User::findOrFail($id);
                return response()->json(['message' => 'Customer Details Fetached successfully', 'record' => new CustomerCollection($user)], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
            
        }
    
    
        /**
         * Update the specified resource in storage.
         */
        public function update(Request $request, $id)
        {
            try {
                $validatedData = $request->validate([
                    'name' => 'required|string|max:100',
                    'email' => 'required|unique:users,email,' . $id . '|max:255',
                    //'phone' => 'sometimes|required|unique:users,phone,' . $id,
                    //'password' => 'sometimes|nullable|string|min:8|confirmed',
                ]);
    
                DB::beginTransaction();
        
                $user = User::findOrFail($id);
                $customer = Customer::where('user_id', $user->id)->firstOrFail();
        
                // Update user data
                $user->name = $request->name;
                $user->email = $request->email;
                if ($request->has('phone')) {
                    $user->phone = $request->phone;
                }
                if ($request->filled('password')) {
                    $user->password = bcrypt($request->password);
                }
                $user->status = $request->status ?? $user->status;
                $user->save();

                $customer->company_name = $request->company_name; 
                $customer->whatsapp_number = $request->whatsapp_number;
                $customer->country_id = $request->country_id;
                $customer->state_id = $request->state_id; 
                $customer->city_id = $request->city_id; 
                $customer->address = $request->address;
                $customer->pincode = $request->pincode;

                $customer->save();
    
                DB::commit();
        
                return response()->json(['message' => 'Customer updated successfully', 'record' => new CustomerCollection($user)], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'User or Customer not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Remove the specified resource from storage.
         */
        public function destroy($id)
        {
            try {
                $user = User::findOrFail($id);
                $user->delete();
                return response()->json(['message' => 'Customer and associated user deleted successfully'], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'User or Customer not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }

        // Profile Pic
        public function profilePic(Request $request){
            try {
                DB::beginTransaction();
                if($request->has('id')){
                    $customer = Customer::where('user_id', $request->id)->firstOrFail();
                }else{
                    $customer = Customer::where('user_id', $request->user()->id)->firstOrFail();
                }
                $customer->profile_pic_id = $request->profile_pic_id ?? $customer->profile_pic_id;
                $customer->save();
                if($request->has('id')){
                    $user = User::findOrFail($request->id);
                }else{
                    $user = User::findOrFail($request->user()->id);
                }
                DB::commit();
                return response()->json(['message' => 'Profile picture is updated successfully', 'record' => new CustomerCollection($user)], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'User or Customer not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }

         // Cover Pic
         public function coverPic(Request $request){
            try {
                DB::beginTransaction();
                if($request->has('id')){
                    $customer = Customer::where('user_id', $request->id)->firstOrFail();
                }else{
                    $customer = Customer::where('user_id', $request->user()->id)->firstOrFail();
                }
                $customer->cover_pic_id = $request->cover_pic_id ?? $customer->cover_pic_id;
                $customer->save();
                if($request->has('id')){
                    $user = User::findOrFail($request->id);
                }else{
                    $user = User::findOrFail($request->user()->id);
                }
                
                DB::commit();
                return response()->json(['message' => 'Cover picture is updated successfully', 'record' => new CustomerCollection($user)], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'User or Customer not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }



        public function showProfile(Request $request)
        {
            try{
                $user = User::findOrFail($request->user()->id);
                return response()->json(['message' => 'Customer Details Fetached successfully', 'record' => new CustomerCollection($user)], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
            
        }
    
    
        /**
         * Update the specified resource in storage.
         */
        public function updateProfile(Request $request)
        {
            try {
                $validatedData = $request->validate([
                    'name' => 'required|string|max:100',
                    'email' => 'required|string|email|max:255|unique:users,email,' . $request->user()->id,
                    'phone' => 'required|string|unique:users,phone,' . $request->user()->id,
                    //'password' => 'sometimes|nullable|string|min:8|confirmed',
                ]);
    
                DB::beginTransaction();
        
                $user = User::findOrFail($request->user()->id);
                $customer = Customer::where('user_id', $user->id)->firstOrFail();
        
                // Update user data
                $user->name = $request->name;
                $user->email = $request->email;
                if ($request->has('phone')) {
                    $user->phone = $request->phone;
                }
                if ($request->filled('password')) {
                    $user->password = bcrypt($request->password);
                }
                $user->save();
                // Customer 
                $customer->company_name = $request->company_name; 
                $customer->whatsapp_number = $request->whatsapp_number;
                $customer->country_id = $request->country_id;
                $customer->state_id = $request->state_id; 
                $customer->city_id = $request->city_id; 
                $customer->address = $request->address;
                $customer->pincode = $request->pincode;

                $customer->save();
    
                DB::commit();
        
                return response()->json(['message' => 'Your profile updated successfully', 'record' => new CustomerCollection($user)], 200);
            }  catch (ValidationException $e) {
                return response()->json([
                    'message' => 'Validation failed',
                    'errors' => $e->errors(),
                ], 422);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'User or Customer not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }

        public function customerCounter(Request $request)
        {
            try{
                $counter = User::where('role','customer')->count();
                return response()->json(['counter' => $counter], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
            
        }
}
