<?php

namespace App\Http\Controllers;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;

use App\Models\Attribute;
use Illuminate\Http\Request;
use App\Http\Resources\Attribute as AttributeCollection;

class AttributeController extends Controller
{
    public function index(Request $request)
        {
            try {

                $limit = $request->query('limit', 10);

                $query = Attribute::query();
                // Get all parameters from the request
                $parameters = $request->all();

            // Loop through each parameter and apply filters to the query
                foreach ($parameters as $key => $value) {
                    // Make sure the key is not a reserved Laravel parameter
                    if ($key !== 'page' && $key !== 'limit') {
                        if (in_array($key, ['attribute_name'])) {
                            // Apply search functionality using 'like' for name, email, and phone
                            $query->where($key, 'like', "%$value%");
                        } else {
                            // Apply exact match for other fields
                            $query->where($key, $value);
                        }
                    }
                }

                $attributes = $query->orderBy('id')->paginate($limit);
                $attributeCollection = AttributeCollection::collection($attributes->items());

                $responseData = [
                    'records' => $attributeCollection,
                    'pagination' => [
                        'current_page' => $attributes->currentPage(),
                        'last_page' => $attributes->lastPage(),
                        'per_page' => $attributes->perPage(),
                        'total' => $attributes->total(),
                    ],
                ];
                
                return response()->json($responseData);
            } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
    
        /**
         * Store a newly created resource in storage.
         */
        public function store(Request $request)
        {
            try {
                $validatedData = $request->validate([
                    'attribute_name' => 'required',
                    'product_id' => 'required',
                ]);
    
                DB::beginTransaction();
                $attribute = Attribute::create([
                    'attribute_name' => $request->attribute_name,
                    'product_id' => $request->product_id,
                    'status' => $request->status ?? true,
                ]);
     
                DB::commit();
                return response()->json(['message' => 'Attribute created successfully', 'record' => new AttributeCollection($attribute)], 201);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Display the specified resource.
         */
        public function show($id)
        {
            try{
                $attribute = Attribute::findOrFail($id);
                return response()->json(['message' => 'Attribute Details Fetached successfully', 'record' => new AttributeCollection($attribute)], 200);
            }catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
            
        }
    
    
        /**
         * Update the specified resource in storage.
         */
        public function update(Request $request, $id)
        {
            try {
                $validatedData = $request->validate([
                    'attribute_name' => 'required',
                    'product_id' => 'required',
                ]);
    
                DB::beginTransaction();
        
                $attribute = Attribute::findOrFail($id); 
                // Update user data

                $attribute->attribute_name = $request->attribute_name;
                $attribute->product_id = $request->product_id;
                $attribute->status = $request->status;
                $attribute->save();
    
                DB::commit();
        
                return response()->json(['message' => 'Attribute updated successfully', 'record' => new AttributeCollection($attribute)], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Attribute not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
    
        /**
         * Remove the specified resource from storage.
         */
        public function destroy($id)
        {
            try {

                $ids = explode(",",$id);
                if (!empty($ids)) {
                    // Fetch the attribute  with the specified IDs
                    $attributes = Attribute::whereIn('id', $ids)->get();
                    
                    // Delete the fetched attribute groups
                    foreach ($attributes as $attribute) {
                        $attribute->delete();
                    }
                }
                return response()->json(['message' => 'Attribute and associated entries deleted successfully'], 200);
            } catch (ModelNotFoundException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Attribute not found'], 404);
            } catch (QueryException $e) {
                DB::rollBack();
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }


        // Other Resources

        public function activeAttribute()
        {
            try {
                $attribute = Attribute::where('status',true)->orderBy('id')->get();
    
                return response()->json(['records' => AttributeCollection::collection($attribute)]);
            } catch (QueryException $e) {
                return response()->json(['message' => 'Database error: ' . $e->getMessage()], 500);
            } catch (\Exception $e) {
                return response()->json(['message' => 'An unexpected error occurred: ' . $e->getMessage()], 500);
            }
        }
}
